/*  WEB-TO-APP BOT – Node.js
    Run: node index.js
*/
const { Telegraf, Markup } = require('telegraf');
const axios = require('axios');
const fs = require('fs');
const path = require('path');
const config = require('./config');

// ---------- helper total user ----------
const userFile = path.join(__dirname, config.totalUserFile || 'users.json');
let userCache = { ids: [] };
if (fs.existsSync(userFile)) userCache = JSON.parse(fs.readFileSync(userFile));
function addUser(id) {
    if (!userCache.ids.includes(id)) {
        userCache.ids.push(id);
        fs.writeFileSync(userFile, JSON.stringify(userCache, null, 2));
    }
}
function totalUsers() { return userCache.ids.length; }

// ---------- bot ----------
const bot = new Telegraf(config.botToken);

// middleware catat user
bot.use((ctx, next) => {
    if (ctx.from) addUser(ctx.from.id);
    return next();
});

// ---------- keyboard ----------
const mainKb = () => Markup.inlineKeyboard([
    [Markup.button.url('📢 Channel', `https://t.me/${config.channelUsername.replace('@', '')}`)],
    [Markup.button.url('👨‍💻 Developer', `https://t.me/${config.developerUsername.replace('@', '')}`)]
]);

// ---------- start ----------
bot.start((ctx) => ctx.reply('🤖 Halo! Saya *Web-to-App Bot*.\nGunakan /menu untuk melihat perintah.', { parse_mode: 'Markdown', ...mainKb() }));

// ---------- menu ----------
bot.command('menu', (ctx) => {
    const txt = `
🤖 *MENU PERINTAH*

┌ *BUAT APLIKASI*
│ 1. /caracreate – baca dulu
│ 2. /toapp \`web,email,namaapk,linklogo,linksplash\`
│ 3. /statusapp \`appId\`
│ 4. /downloadapp \`appId\`
└──────────────

┌ *LAINNYA*
│ • /report \`pesan\` – lapor bug
│ • /total – total pengguna
│ • /menu – tampilkan ini lagi
└──────────────

📢 Channel : ${config.channelUsername}
👨‍💻 Dev : ${config.developerUsername}
`.trim();
    ctx.reply(txt, { parse_mode: 'Markdown', disable_web_page_preview: true, ...mainKb() });
});

// func anti error website 
function esc(str) {
    return str.replace(/[_*[\]()~`>#+\-=|{}.!\\]/g, '\\$&');
}

// ---------- caracreate ----------
bot.command('caracreate', (ctx) => {
    const text = esc(`📖 *Cara Membuat Aplikasi*

1. Siapkan: link website, email, nama aplikasi, link logo, link splash
2. Upload logo & splash ke [https://catbox.moe](https://catbox.moe) atau pakai @KhafaBeta_bot
3. Ketik:
\`/toapp https://web.com,email@gmail.com,NamaApk,https://i.catbox.moe/abc.jpg,https://i.catbox.moe/def.jpg\`
4. Tunggu 5–20 menit
5. Cek status: \`/statusapp appId\`
6. Kalau sukses: \`/downloadapp appId\``);

    ctx.reply(text, { parse_mode: 'MarkdownV2', ...mainKb() });
});

// ---------- toapp ----------
bot.command('toapp', async (ctx) => {
    const args = ctx.message.text.split(' ').slice(1).join(' ').split(',').map(x => x.trim());
    if (args.length !== 5) return ctx.reply('❗ Format salah.\nContoh:\n`/toapp web.com,email@gmail.com,NamaApk,linklogo,linksplash`', { parse_mode: 'Markdown' });

    const [url, email, appName, appIcon, splashIcon] = args;
    try {
        const { data } = await axios.get('https://api.resellergaming.my.id/tools/toapp/create', {
            params: { url, email, appName, appIcon, splashIcon }
        });
        if (data.status) {
            ctx.reply(`✅ *Berhasil memulai pembuatan!*\nApp ID: \`${data.appId}\`\nDetail: ${data.message}\n\nCek status: /statusapp ${data.appId}`, { parse_mode: 'Markdown', ...mainKb() });
        } else {
            ctx.reply('❌ Gagal memulai pembuatan.');
        }
    } catch (e) {
        console.error(e);
        ctx.reply('❌ Kesalahan server.');
    }
});

// ---------- statusapp ----------
bot.command('statusapp', async (ctx) => {
    const appId = ctx.message.text.split(' ')[1];
    if (!appId) return ctx.reply('❗ Gunakan: `/statusapp appId`', { parse_mode: 'Markdown' });
    try {
        const { data } = await axios.get('https://api.resellergaming.my.id/tools/toapp/status', { params: { appId } });
        if (data.status) {
            const st = data.buildStatus;
            const emoji = st === 'success' ? '✅' : st === 'failed' ? '❌' : '⏳';
            ctx.reply(`${emoji} *Status:* ${st}\nApp ID: \`${data.appId}\``, { parse_mode: 'Markdown', ...mainKb() });
        } else {
            ctx.reply('❌ App ID tidak ditemukan.');
        }
    } catch (e) {
        console.error(e);
        ctx.reply('❌ Kesalahan cek status.');
    }
});

// ---------- downloadapp ----------
bot.command('downloadapp', async (ctx) => {
    const appId = ctx.message.text.split(' ')[1];
    if (!appId) return ctx.reply('❗ Gunakan: `/downloadapp appId`', { parse_mode: 'Markdown' });
    try {
        const { data } = await axios.get('https://api.resellergaming.my.id/tools/toapp/download', { params: { appId } });
        if (data.status) {
            const txt = `
✅ *Aplikasi siap diunduh*

Nama: ${data.appName}
Package: ${data.packageName}

📥 [Download APK](${data.downloadUrl})
📦 [AAB File](${data.aabFile})
🔐 [KeyStore](${data.keyFile})

\`StorePass: ${data.storePass}\`
\`KeyPass: ${data.keyPass}\`
`.trim();
            ctx.reply(txt, { parse_mode: 'Markdown', disable_web_page_preview: true, ...mainKb() });
        } else {
            ctx.reply('❌ App ID tidak ditemukan / build belum selesai.');
        }
    } catch (e) {
        console.error(e);
        ctx.reply('❌ Kesalahan download.');
    }
});

// ---------- report ----------
bot.command('report', async (ctx) => {
    const pesan = ctx.message.text.split(' ').slice(1).join(' ');
    if (!pesan) return ctx.reply('❗ Gunakan: `/report pesan laporan`', { parse_mode: 'Markdown' });
    try {
        await axios.post(`https://api.telegram.org/bot${config.reportBotToken}/sendMessage`, {
            chat_id: config.reportChatId,
            text: `📩 *Laporan Baru*\nDari: @${ctx.from.username || ctx.from.id}\nPesan: ${pesan}`,
            parse_mode: 'Markdown'
        });
        ctx.reply('✅ Laporan terkirim. Terima kasih!');
    } catch (e) {
        console.error(e);
        ctx.reply('❌ Gagal kirim laporan.');
    }
});

// ---------- total ----------
bot.command('total', (ctx) => {
    ctx.reply(`👥 Total pengguna sejak bot start: *${totalUsers()}* orang`, { parse_mode: 'Markdown', ...mainKb() });
});

// ---------- launch ----------
bot.launch();
console.log('✅ Bot Web-to-App berjalan...');

// graceful stop
process.once('SIGINT', () => bot.stop('SIGINT'));
process.once('SIGTERM', () => bot.stop('SIGTERM'));